<?php
header('Content-Type: application/json');

$config = require_once __DIR__ . '/donate/config.php';
$type = strtolower($config['ServerFiles'] ?? 'louis');

if (!isset($config['mssql_host'], $config['mssql_db'], $config['mssql_user'], $config['mssql_password'])) {
    die(json_encode(["error" => "❌ Missing MSSQL config variables in donate/config.php"]));
}

try {
    $pdo = new PDO(
        "dblib:host={$config['mssql_host']};dbname={$config['mssql_db']};charset=UTF-8",
        $config['mssql_user'],
        $config['mssql_password']
    );
    $pdo->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
} catch (PDOException $e) {
    die(json_encode(["error" => "❌ DBLIB connection failed", "details" => $e->getMessage()]));
}

function fetch_and_save($pdo, $sql, $filename, $indexBy = null) {
    try {
        $stmt = $pdo->query($sql);
        $data = [];

        while ($row = $stmt->fetch(PDO::FETCH_ASSOC)) {
            foreach ($row as $key => $value) {
                if (is_resource($value)) {
                    $row[$key] = base64_encode(stream_get_contents($value));
                } elseif ($value instanceof DateTime) {
                    $row[$key] = $value->format('Y-m-d H:i:s');
                } elseif (is_string($value)) {
                    // ✅ Unicode поддерживается — не трогаем
                    $row[$key] = $value;
                }
            }

            if ($indexBy && isset($row[$indexBy])) {
                $id = strtolower(trim($row[$indexBy]));
                $data[$id] = $row;
            } else {
                $data[] = $row;
            }
        }

        file_put_contents(
            __DIR__ . "/$filename",
            json_encode($data, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT)
        );

        return count($data);
    } catch (PDOException $e) {
        file_put_contents(__DIR__ . "/$filename.error.log", $e->getMessage());
        return 0;
    }
}

// Серверно-зависимые запросы
switch ($type) {
    case 'mspro':
    case 'louis':
        $char_query = "SELECT Name, Class, cLevel, ResetCount AS ResetCount, MasterResetCount AS MasterResetCount, Kills, Deads FROM Character";
        $cashshop_query = "SELECT AccountID, WCoinC, WCoinP, GoblinPoint FROM CashShopData";
        $memb_info_query = "SELECT memb_guid, memb___id, AccountLevel, AccountExpireDate FROM MEMB_INFO";
        $memb_stat_query = "SELECT memb___id, ConnectStat, OnlineHours FROM MEMB_STAT";
        $accountchar_id = "Number";
        $guild_query = "
            SELECT GM.G_Name, G.G_Master, GM.Name, GM.G_Status 
            FROM GuildMember GM 
            JOIN Guild G ON GM.G_Name = G.G_Name
        ";
        break;

    case 'mudevs':
        $char_query = "
            SELECT 
                C.Name, 
                C.Class, 
                C.cLevel, 
                C.ResetCount AS ResetCount,
                C.MasterResetCount AS MasterResetCount,
                ISNULL(D.Killed, 0)   AS Kills,
                ISNULL(D.Murdered, 0) AS Deads
            FROM Character C
            LEFT JOIN DeathRecords D ON D.Name = C.Name
        ";
        $cashshop_query = "SELECT AccountID, WCoinC, WCoinP, GoblinPoint FROM CashShopData";
        $memb_info_query = "SELECT memb_guid, memb___id FROM MEMB_INFO";
        $memb_stat_query = "SELECT memb___id, ConnectStat, OnlineHours FROM MEMB_STAT";
        $accountchar_id = "AccountIndex";
        $guild_query = "
            SELECT GM.G_Name, G.G_Master, GM.Name, GM.G_Status 
            FROM GuildMember GM 
            JOIN Guild G ON GM.G_Name = G.G_Name
        ";
        break;

    case 'ignc':
        $char_query = "SELECT Name, Class, cLevel, RESETS AS ResetCount, GrandResets AS MasterResetCount, WinDuels AS Kills, LoseDuels AS Deads FROM Character";
        $cashshop_query = "SELECT AccountID, WCoin, GoblinPoint FROM T_InGameShop_Point";
        $memb_info_query = "SELECT memb_guid, memb___id, VipType, VipStart, VipDays FROM MEMB_INFO";
        $memb_stat_query = "SELECT memb___id, ConnectStat, ConnectTM FROM MEMB_STAT";
        $accountchar_id = "Number";
        $guild_query = "
            SELECT GM.G_Name, G.G_Master, GM.Name, GM.G_Status 
            FROM GuildMember GM 
            JOIN Guild G ON GM.G_Name = G.G_Name
        ";
        break;

    case 'ssemu':
        $char_query = "
            SELECT 
                Name, 
                Class, 
                cLevel, 
                ResetCount AS ResetCount, 
                MasterResetCount AS MasterResetCount,
                0 AS Kills,
                0 AS Deads
            FROM Character
        ";
        $cashshop_query = null; // нет CashShopData
        $memb_info_query = "SELECT memb_guid, memb___id, AccountLevel, AccountExpireDate FROM MEMB_INFO";
        $memb_stat_query = "SELECT memb___id, ConnectStat, OnlineHours FROM MEMB_STAT";
        $accountchar_id = "Number";
        $guild_query = "
            SELECT GM.G_Name, G.G_Master, GM.Name 
            FROM GuildMember GM 
            JOIN Guild G ON GM.G_Name = G.G_Name
        ";
        break;

    default:
        die(json_encode(["error" => "❌ Unsupported server type: $type"]));
}

// 📦 Единый экспорт
$result = [
    "characters"       => fetch_and_save($pdo, $char_query, "characters-cache.json", "Name"),
    "memb_stat"        => fetch_and_save($pdo, $memb_stat_query, "memb_stat.json", "memb___id"),
    "guildmember"      => fetch_and_save($pdo, $guild_query, "guildmember.json", "Name"),
    "accountcharacter" => fetch_and_save($pdo, "
        SELECT $accountchar_id AS Number, Id, GameID1, GameID2, GameID3, GameID4, GameID5, GameIDC 
        FROM AccountCharacter
    ", "accountcharacter.json", "Id"),
    "cashshopdata"     => $cashshop_query 
        ? fetch_and_save($pdo, $cashshop_query, "cashshopdata.json", "AccountID")
        : [], // если таблицы нет — вернуть пустой массив
    "memb_info"        => fetch_and_save($pdo, $memb_info_query, "memb_info.json", "memb___id")
];

echo json_encode([
    "success"   => true,
    "exported"  => $result,
    "timestamp" => date("Y-m-d H:i:s")
], JSON_PRETTY_PRINT);
